/******************************************
DUCK PUNT
Copyright (C) 2005 Geoffrey M. Draper

This file is part of Duck Punt.
Duck Punt is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************/

import javax.swing.ImageIcon;

public class Duck extends Sprite
{
	private ImageIcon sitting_image[];
	private ImageIcon flying_image[];
	private final int flying_frames = 8;
	private float yardage;
	private boolean interception;

	//used for animation
	private float radians;			//current radian value for the duck (upon which punt height is based)
	private float radial_increment;	//value by which radians is increased at each iteration
	private float y_boost;			//random value to multiply duck's y-value by (to make punt higher)
	private int punt_length;		//how many yards the punt ought to go
	private int flyingX;		//bogus X value used while duck is flying, for mathematical convenience
	private int target_yardage;	//yardage the player will achieve if current punt lands unscathed
	private int num_frames_for_intercepter_to_run_before_grabbing_duck;	//how many iterations for intercepter to run from side of screen to duck
	private int step_val;		//number of sideways (X) pixels duck moves at each iteration
	private float yardage_step_val;	//number of yards duck moves at each iteration
	private int num_frames_to_fly_before_intercepter_appears;	//how many iterations before we draw intercepter

	public Duck(int sv)
	{
		x=900;y=-5;
		flyingX=x;
		step_val = sv;
		yardage_step_val = ((float)step_val / (float)DuckPunt.pixelsToYard);
		interception = false;
		radians=0;
		
		//load graphics
		sitting_image = new ImageIcon[2];
		sitting_image[0] = new ImageIcon("data/images/right_duck.gif");
		sitting_image[1] = new ImageIcon("data/images/left_duck.gif");
		flying_image = new ImageIcon[flying_frames];
		for (int i=0; i<flying_frames; i++)
		{
			flying_image[i] = new ImageIcon("data/images/flying_" + String.valueOf(i) + ".gif");
		}
		
		//create a new random number generator
		Math.random();
	}

	public void setYardage(int yrd)
	{
		yardage = (float)yrd;
	}

	public int getYardage()
	{
		return (int)yardage;
	}

	public float getRealYardage()
	{
		return yardage;
	}

	public int getX()
	{
		//overloaded for flying sake
		//(there may be a better way to do this!)
		return flyingX;
	}

	public int getPuntLength()
	{
		return punt_length;
	}
	
	public boolean isAlmostReadyToIntercept()
	{
		//this function returns TRUE as soon as we're ready
		//for the intercepter to appear on-screen
		
		return (interception && ((--num_frames_to_fly_before_intercepter_appears) == 0));
	}

	public boolean isReadyToIntercept()
	{
		//this function returns TRUE as soon as the duck
		//is close enough to the intercepter to get caught
		
		return ((--num_frames_for_intercepter_to_run_before_grabbing_duck) == 0);
	}

	public boolean willBeIntercepted()
	{
		return interception;
	}

	public void moveRelativeToPlayer()
	{
		x -= step_val;
		flyingX = x;
	}

	public void setupPunt(int diff, int range, int default_length, int screen_width)
	{
		//optionally, the default_length parameter can override the punt_length value
		float max_punt = 30;
		float maria = Math.abs(max_punt + (diff-range/2)*(diff-range/2) / (-4 * max_punt));
		float random = (float)Math.random();
		float interception_chance = (float)Math.random();
		float computed_radians;		//what angle is the duck at when at intercepter's arm-height?
		
		num_frames_for_intercepter_to_run_before_grabbing_duck = 0;	//bogus default value
		computed_radians = 0;	//bogus default value (keep compiler happy)
		interception = false;
		
		//if we were passed a default length for this punt, use it.
		if (default_length > 0)
		{
			punt_length = default_length;
		//otherwise, create one pseudo-randomly
		} else {			
			punt_length = 2 + (int)(maria * random);
			
			//will this punt be intercepted by the opposing team?
			if (interception_chance > 0.7 && yardage > 30.0 && punt_length > 10.0)
				//(don't intercept if too close to touchdown or if kick is too short
			{
				interception = true;
			}
		}
		
		target_yardage = Math.max(0,getYardage() - punt_length);

		y_boost = 100 + (range-diff) * (float)(0.5 + Math.random());
		
		if (interception)
		{
			num_frames_for_intercepter_to_run_before_grabbing_duck = (screen_width - x) / step_val;
			//now, at what radian value will the duck be at arm-height?
			//find that radian value, and see at what iteration# the duck will be at that height
			//then subtract num_frames_blah_blah from that value and start the interceptor THEN.
			float arm_height = 60;
			float computed_height = arm_height / y_boost;	//normalize to between 0 and 1
			
			//we subtract this from pi so that we get the radian value for when the duck is falling, not rising
			computed_radians = (float)(Math.PI - Math.asin((double)computed_height));
		}

		radians = 0;
		radial_increment = (float) ((Math.PI / (double)punt_length) / ((double)DuckPunt.pixelsToYard / (double)step_val));
		
		if (interception)
		{
			int num_frames_to_reach_arm_height = (int)(computed_radians / radial_increment);
			num_frames_to_fly_before_intercepter_appears = num_frames_to_reach_arm_height - num_frames_for_intercepter_to_run_before_grabbing_duck;
			//prevent impossible interceptions
			if (num_frames_to_fly_before_intercepter_appears < 1) {interception = false;}
		}
	}

	public boolean animate()
	{
		radians += radial_increment;
		y = (int)(Math.sin(radians) * y_boost);
		flyingX -= step_val;
		yardage -= yardage_step_val;

		//having the duck spin around each frame gets kind of dizzying,
		//so I introduce this artificial slowdown to make it spin only
		//once every four frames, which looks nicer I think.
		if (frameMod++ % 4 == 0)
		{
			current_image = flying_image[(int)(Math.random() * (double)flying_frames)];
		}

		//return false if this is the last frame
		if (getYardage() <= target_yardage) {
			return false;
		} else {
			return true;
		}
	}

	public void returnToHomePosition(int player, int window_width)
	{
		x=100+(window_width/2)+(int)(Math.random() * (double)400.0);
		y=-5;
		flyingX=x;
		frameMod = 0;
		interception = false;
		current_image = sitting_image[player];
	}
	
	//move the duck over more, to give new players time
	//to figure out the user-interface.
	public void padForBeginners(int padding)
	{
		x += padding;
		flyingX = x;
	}
	
	public static void say(String s)
	{
		System.out.println(s);
	}

}
