package projman;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/*
 * I intentionally did not make this class serializable, since there's
 * no need to save/restore diffs.  They ought to be computed on the fly.
 * 
 * FIXME: I had to make it implement Serializable because it kept on
 * throwing errors otherwise. But it shouldn't be this way. Figure out
 * what serialized object has a reference to DiffWard, and fix it!
 */
public class DiffWard implements Serializable {

	public enum Status {
		SAME,
		OLD,
		NEW
	}

	private Map<Skill, Status> skills;
	private Map<Project, DiffProject> projects;
	private Map<Resource, DiffResource> resources;
	private Map<String, Resource> resourceNames;
	private Map<String, Project> projectNames;
	private Ward ward1, ward2;

	public DiffWard(Ward ward1, Ward ward2) {
		skills = new HashMap<Skill, Status>();
		projects = new HashMap<Project, DiffProject>();
		resources = new HashMap<Resource, DiffResource>();
		resourceNames = new HashMap<String, Resource>();
		projectNames = new HashMap<String, Project>();
		this.ward1 = ward1;
		this.ward2 = ward2;
		recomputeDifferences();
	}

	public boolean add(Entity entity, Status status) {
		boolean result = false;
		if (entity instanceof Project) {
			projects.put((Project)entity, new DiffProject((Project)entity, status));
			projectNames.put(entity.getName(), (Project)entity);
			result = true;
		} else if (entity instanceof Resource) {
			resources.put((Resource)entity, new DiffResource((Resource)entity, status));
			resourceNames.put(entity.getName(), (Resource)entity);
			result = true;
		} else if (entity instanceof Skill) {
			skills.put((Skill)entity, status);
			result = true;
		}
		return result;
	}

	public void recomputeDifferences() {
		//Main.say("Starting to compute diffs...");
		//long startingTime = System.currentTimeMillis();
		List<Project> unmatchedProjects1 = new LinkedList<Project>();
		List<Project> unmatchedProjects2 = new LinkedList<Project>();
		List<Resource> unmatchedResources1 = new LinkedList<Resource>();
		List<Resource> unmatchedResources2 = new LinkedList<Resource>();
		List<Skill> unmatchedSkills1 = new LinkedList<Skill>();
		List<Skill> unmatchedSkills2 = new LinkedList<Skill>();
		skills.clear();
		projects.clear();
		resources.clear();

		for (Project p : ward1.getAllProjects()) {
			Team c = p.getTeam();
			if (c==null) c = new Team();
			unmatchedProjects1.add(p);
		}
		for (Project p : ward2.getAllProjects()) {
			Team c = p.getTeam();
			if (c==null) c = new Team();
			unmatchedProjects2.add(p);
		}
		for (Resource r : ward1.getAllResources()) {
			unmatchedResources1.add(r);
		}
		for (Resource r : ward2.getAllResources()) {
			unmatchedResources2.add(r);
		}
		for (Skill r : ward1.getAllSkills()) {
			unmatchedSkills1.add(r);
		}
		for (Skill r : ward2.getAllSkills()) {
			unmatchedSkills2.add(r);
		}

		//find diffs in the projects
		for (Project proj1 : ward1.getAllProjects()) {
			for (Project proj2 : ward2.getAllProjects()) {
				//Main.say("comparing " + proj1.getName() + " to " + proj2.getName());
				if (proj1.equals(proj2)) {
					//Main.say("both charts have a project called: " + proj1.getName());
					unmatchedProjects1.remove(proj1);
					unmatchedProjects2.remove(proj2);
					add(proj2, Status.SAME);
					break;
				}
			}
		}
		for (Project p : unmatchedProjects1) {
			//Main.say("chart 1 has a project called " + p.getName() + " that does not exist in chart 2.");
			add(p, Status.OLD);
		}
		for (Project p : unmatchedProjects2) {
			//Main.say("chart 2 has a project called " + p.getName() + " that does not exist in chart 1.");
			add(p, Status.NEW);
		}

		//find diffs in the resources
		for (Resource res1 : ward1.getAllResources()) {
			for (Resource res2 : ward2.getAllResources()) {
				//Main.say("comparing " + proj1.getName() + " to " + proj2.getName());
				if (res1.equals(res2)) {
					//Main.say("both charts have a resource called: " + res1.getName());
					unmatchedResources1.remove(res1);
					unmatchedResources2.remove(res2);
					add(res2, Status.SAME);
					break;
				}
			}
		}
		for (Resource p : unmatchedResources1) {
			//Main.say("chart 1 has a resource called " + p.getName() + " that does not exist in chart 2.");
			add(p, Status.OLD);
		}
		for (Resource p : unmatchedResources2) {
			//Main.say("chart 2 has a resource called " + p.getName() + " that does not exist in chart 1.");
			add(p, Status.NEW);
		}

		//find diffs in the skills
		for (Skill res1 : ward1.getAllSkills()) {
			for (Skill res2 : ward2.getAllSkills()) {
				if (res1.equals(res2)) {
					//Main.say("both charts have a skill called: " + res1.getName());
					unmatchedSkills1.remove(res1);
					unmatchedSkills2.remove(res2);
					add(res2, Status.SAME);
					break;
				}
			}
		}
		for (Skill p : unmatchedSkills1) {
			//Main.say("Chart 1 has a skill called " + p.getName() + " that does not exist in chart 2.");
			add(p, Status.OLD);
		}
		for (Skill p : unmatchedSkills2) {
			//Main.say("Chart 2 has a skill called " + p.getName() + " that does not exist in chart 1.");
			add(p, Status.NEW);
		}

		//look for differences within project teams
		for (DiffProject dp : projects.values()) {
			dp.computeDifferences(ward1, ward2);
		}

		//check to see which skills these resources had in chart 1 versus chart 2.
		//look for differences within project teams
		for (DiffResource dr : resources.values()) {
			dr.computeDifferences(ward1, ward2);
		}

		//test();

		//Main.say("Done computing diffs. Time: " + (System.currentTimeMillis() - startingTime) +  " ms.");

	}
	
	public List<DiffProject> getAllProjects() {
		List<DiffProject> sortedProjects = new ArrayList<DiffProject>(projects.values());
		Collections.sort(sortedProjects);
		return sortedProjects;
	}
	
	public DiffProject getDiffProject(Project p) {
		return projects.get(p);
	}
	
	public List<DiffResource> getAllResources() {
		List<DiffResource> sortedResources = new ArrayList<DiffResource>(resources.values());
		Collections.sort(sortedResources);
		return sortedResources;
	}
	
	public DiffResource getDiffResource(Resource r) {
		return resources.get(r);
	}
	
	public List<Skill> getAllSkills() {
		List<Skill> sortedSkills = new ArrayList<Skill>(skills.keySet());
		Collections.sort(sortedSkills);
		return sortedSkills;
	}
		
	public Status getStatus(Entity k) {
		Status result = null;
		if (k instanceof Skill) {
			result = skills.get(k);
		}
		if (k instanceof Project) {
			DiffProject dp = projects.get(k);
			result = dp.getProjectStatus();
		}
		if (k instanceof Resource) {
			Resource res = resourceNames.get(k.getName());
			DiffResource dr = resources.get(res);
			result = dr.getResourceStatus();
		}
		return result;
	}
	
	private void test() {
		//This is just an sanity-checker. You can delete it or comment it out
		//once you're satisfied that the algorithm is working.
		for (DiffProject dc : projects.values()) {
			Project p = dc.getProject();
			Status s = dc.getProjectStatus();
			if (s == Status.OLD) {
				Main.say("Project " + p.getName() + " existed in Chart #1, but not Chart #2.");
				for (String k : dc.getAllAssociatedSkills()) {
					Main.say("Project " + p.getName() + " required " + dc.getCountForSkill(k, s) + " " + k + "s.");
				}
				Team workers = p.getTeam();
				if (workers != null) {
					for (Resource r : workers.getResources()) {
						Main.say("Resource " + r.getName() + " worked on project " + p.getName() + " in Chart #1.");
					}
				}
			}
			if (s == Status.NEW) {
				Main.say("Project " + p.getName() + " exists in Chart #2, but not Chart #1.");
				for (String k : dc.getAllAssociatedSkills()) {
					Main.say("Project " + p.getName() + " requires " + dc.getCountForSkill(k, s) + " " + k + "s.");
				}
				Team workers = p.getTeam();
				if (workers != null) {
				for (Resource r : workers.getResources()) {
					Main.say("Resource " + r.getName() + " works on project " + p.getName() + " in Chart #2.");
				}
				}
			}
			if (s == Status.SAME) {
				Main.say("Project " + p.getName() + " existed in both charts.");
				for (Resource r : dc.getResources()) {
					s = dc.getResourceStatus(r);
					if (s == Status.OLD) {
						Main.say("Resource " + r.getName() + " was assigned to " + p.getName() + " in Chart #1, but not Chart #2.");
					}
					if (s == Status.NEW) {
						Main.say("Resource " + r.getName() + " was assigned to " + p.getName() + " in Chart #2, but not Chart #1.");
					}
					if (s == Status.SAME) {
						Main.say("Resource " + r.getName() + " was assigned to " + p.getName() + " in both charts.");
					}			
				}
				for (String k : dc.getAllAssociatedSkills()) {
					int count1 = dc.getCountForSkill(k, Status.OLD);
					int count2 = dc.getCountForSkill(k, Status.NEW);
					if (count1==count2) {
						Main.say("Project " + p.getName() + " required " + count1 + " " + k + "s in both charts.");
					} else {
						Main.say("Project " + p.getName() + " required " + count1 + " " + k + "s in Chart #1.");
						Main.say("Project " + p.getName() + " requires " + count2 + " " + k + "s in Chart #2.");
					}
				}
			}
		}
		
		for (DiffResource dc : resources.values()) {
			Resource p = dc.getResource();
			Status s = dc.getResourceStatus();
			if (s == Status.OLD) {
				Main.say("Resource " + p.getName() + " existed in Chart #1, but not Chart #2.");
			}
			if (s == Status.NEW) {
				Main.say("Resource " + p.getName() + " existed in Chart #2, but not Chart #1.");
			}
			if (s == Status.SAME) {
				Main.say("Resource " + p.getName() + " existed in both charts.");
				for (String k : dc.getSkills()) {
					s = dc.getSkillStatus(k);
					if (s == Status.OLD) {
						Main.say("Resource " + p.getName() + " had the skill " + k + " in Chart #1, but not Chart #2.");
					}
					if (s == Status.NEW) {
						Main.say("Resource " + p.getName() + " had the skill " + k + " in Chart #2, but not Chart #1.");
					}
					if (s == Status.SAME) {
						Main.say("Resource " + p.getName() + " had the skill " + k + " in both charts.");
					}	
				}
			}

		}
		
		for (Skill k : skills.keySet()) {
			Status s = skills.get(k);
			if (s == Status.OLD) {
				Main.say("Chart 1 has a skill called " + k.getName() + " that does not exist in chart 2.");
			}
			if (s == Status.NEW) {
				Main.say("Chart 2 has a skill called " + k.getName() + " that does not exist in chart 1.");
			}
			if (s == Status.SAME) {
				Main.say("Skill " + k.getName() + " exists in both charts.");
			}	

		}
	}


}
