package projman;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Map;
import java.util.TreeMap;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;

public class EditResourceDialog extends JDialog implements ActionListener {
	
	private Resource brother;
	private JTextField nameTextField;
	private JSlider capacitySlider;
	private Map<Team, JSpinner> proportions;
	private static final long serialVersionUID = 42L;
	
	public EditResourceDialog(Dialog parent, boolean existing, Resource guy) {
		super(parent, true);
		init(existing, guy, parent);
	}
	
	public EditResourceDialog(Frame parent, boolean existing, Resource guy) {
		super(parent, true);
		init(existing, guy, parent);
	}
	
	private void init(boolean existing, Resource guy, Component parent) {
		if (existing) {
			setTitle(Main.messages.getString(Constants.EDIT_RESOURCE_DIALOG_TITLE));
		} else {
			setTitle(Main.messages.getString(Constants.ADD_NEW_RESOURCE_DIALOG_TITLE));
		}
		this.brother = guy;
		proportions = new TreeMap<Team, JSpinner>();
		buildGUI(existing);
		setResizable(false);
		pack();
		
		//position this dialog in the middle of its parent window
		Rectangle parentBounds = parent.getBounds();
		Dimension mySize = getSize();
		setLocation(parentBounds.x + parentBounds.width/2 - mySize.width/2,
				parentBounds.y + parentBounds.height/2 - mySize.height/2);

		
		//treat a window-closing event as if the user
		//had clicked the "cancel" button.
		addWindowListener(new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent event) {
				handleActionEvent(Constants.CANCEL_BUTTON_LABEL);
			}
		});

		setVisible(true);

	}
	
	private void buildGUI(boolean editExisting) {
		setLayout(new BoxLayout(getContentPane(), BoxLayout.PAGE_AXIS));
		JPanel namePanel = new JPanel();
		JLabel nameLabel = new JLabel(Main.messages.getString(Constants.ENTER_RESOURCE_NAME_DIALOG_LABEL));
		nameTextField = new JTextField();
		if (!editExisting) {
			nameTextField.setText(Main.messages.getString(Constants.DEFAULT_RESOURCE_NAME));
			nameTextField.select(0, Main.messages.getString(Constants.DEFAULT_RESOURCE_NAME).length());
		} else {
			nameTextField.setText(brother.getName());
		}
		namePanel.add(nameLabel);
		namePanel.add(nameTextField);
		
		JPanel capacityPanel = new JPanel();
		capacitySlider = new JSlider();
		capacitySlider.setLabelTable(capacitySlider.createStandardLabels(25));
		capacitySlider.setPaintLabels(true);
		capacitySlider.setValue((int)(100*brother.getCapacity()));
		capacityPanel.add(new JLabel("Capacity: "));
		capacityPanel.add(capacitySlider);
		
		JPanel workPanel_aux = new JPanel();
		workPanel_aux.setBorder(BorderFactory.createEmptyBorder(0,10,10,10));
		Box workPanel = new Box(BoxLayout.PAGE_AXIS);
		workPanel.setBorder(BorderFactory.createLineBorder(Color.BLACK));
		for (Team comp : brother.getCompanionships()) {
			if (comp.getProject() == null) continue;
			JPanel projectPanel = new JPanel();			
			JLabel label = new JLabel(comp.getProject().getName());
			SpinnerModel spinnerModel =
				new SpinnerNumberModel(
						(int)(100*brother.getPercentageOfTimeSpentOnThisProject(comp)), //initial value
						0, //min
						100, //max
						1);  //step
			JSpinner percentWidget = new JSpinner(spinnerModel);
			projectPanel.add(label);
			projectPanel.add(percentWidget);
			proportions.put(comp, percentWidget);
			workPanel.add(projectPanel);
		}
		workPanel_aux.add(workPanel);
		
		JPanel okCancelPanel = new JPanel();
		okCancelPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
		JButton okButton = new JButton(Main.messages.getString(Constants.OK_BUTTON_LABEL));
		getRootPane().setDefaultButton(okButton);
		JButton cancelButton = new JButton(Main.messages.getString(Constants.CANCEL_BUTTON_LABEL));
		okCancelPanel.add(okButton);
		okCancelPanel.add(cancelButton);
		
		okButton.setActionCommand(Constants.OK_BUTTON_LABEL);
		cancelButton.setActionCommand(Constants.CANCEL_BUTTON_LABEL);
		okButton.addActionListener(this);
		cancelButton.addActionListener(this);
		
		add(namePanel);
		add(capacityPanel);
		
		//only show the proportions box if the resource
		//is assigned to at least one project.
		if (proportions.values().size() > 0) add(workPanel_aux);
		
		add(okCancelPanel);
	}

	public void actionPerformed(ActionEvent e) {
		String command = e.getActionCommand();
		handleActionEvent(command);
	}
	
	private void handleActionEvent(String command) {
		if (command.equals(Constants.CANCEL_BUTTON_LABEL)) {
			brother.setChanged(false);
			setVisible(false);
		}
		if (command.equals(Constants.OK_BUTTON_LABEL)) {
			//check that percentages add up to 100
			int sum = 0;
			for (JSpinner spinner : proportions.values()) {
				sum += (Integer)spinner.getValue();
				//Main.say("spinner value: " + spinner.getValue());
			}
			if (proportions.values().size() > 0 && (sum < 99 || sum > 100)) {
				JOptionPane.showMessageDialog(this,
						"The percentages do not add up to 100; please adjust.",
						"Invalid proportions",
						JOptionPane.ERROR_MESSAGE);
			} else {
				brother.setName(nameTextField.getText());
				brother.setCapacity(capacitySlider.getValue()*0.01);
				brother.setChanged(true);
				for (Team comp : proportions.keySet()) {
					brother.setPercentageOfTimeSpentOnThisProject(comp,
							((Integer)proportions.get(comp).getValue())/100.0);
				}
				setVisible(false);
			}
		}
		
	}
	
	public Resource getResource() {
		return brother;
	}
}
