package projman;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.Point2D;
import java.io.Serializable;

import projman.DiffWard.Status;

public abstract class NameCard implements Serializable, ActionListener {
	protected Rectangle geometry;
	private Entity entity;
	final static Dimension MIN_SIZE = new Dimension (100,30);
	static final int buffer = 10;
	protected Point textPosition;
	protected Point centerOfGravity;
	private boolean initialized;
	protected boolean ghost;
	private Point prevPosition;
	protected Status diffStatus;
	protected DiffWard diffWard;
	private int textHeight;
	private boolean flashMode;//TODO maybe move this to WhiteboardCard?
	private static final long serialVersionUID = 42L;
	static final int ROUND_RECT_DIAMETER = 20;
	static final int SIZE_OF_PLUS_SIGN = 15;
	//TODO move these static final vars to Config.

	public NameCard(Entity entity) {
		this.entity = entity;
		geometry = new Rectangle(MIN_SIZE.width, MIN_SIZE.height);
		textPosition = new Point();
		centerOfGravity = new Point();
		prevPosition = new Point();
		initialized = false;
		ghost = false;
		diffWard = null;
		diffStatus = Status.SAME;
		flashMode = false;
	}
	
	public NameCard(Entity entity, Point loc, Graphics2D g) {
		this(entity);
		centerOfGravity.setLocation(loc);
		initialize(g);
		//Main.say("NameCard: " + getName() + " created as" + geometry);
	}
	
	public void scale(float d) {
		geometry.width *= d;
		geometry.height *= d;
	}
	
	public void setSize(final Dimension d) {
		geometry.setSize(d);
	}
	
	public void setPosition(final Point p) {
		if (initialized) {
			Point delta = new Point(p.x-centerOfGravity.x,
					p.y-centerOfGravity.y);
			drag(delta);
		} else {
			centerOfGravity.setLocation(p);
		}
	}
	
	public void setBottomLeftPosition(final Point loc) {
		Point p = new Point(loc);
		p.translate(geometry.width/2, -geometry.height/2);
		setPosition(p);
	}

	public void setTopLeftPosition(final Point loc) {
		Point p = new Point(loc);
		p.translate(geometry.width/2, geometry.height/2);
		setPosition(p);
	}
	
	public Point getPosition() {
		return centerOfGravity;
	}
	
	public boolean containsPoint(final Point p) {
		return geometry.contains(p);
	}
	
	protected boolean isInitialized() {
		return initialized;
	}
	
	public Rectangle getBounds() {
		return geometry;
	}
	
	private Color getCardColor() {
		Color result = Config.getColorForEntity(getEntity(), diffStatus);
		//make up for entities that are old wrt their projects, but not
		//old to the ward.
		if (diffStatus == Status.OLD && diffWard.getStatus(entity) != diffStatus) {
			result = Config.getColorForEntity(getEntity(), Status.SAME);
		}
		return result;
	}
	
	private void initialize(Graphics2D g) {
		if (g != null) {
			initialized = true;
			recalculateSize(g);
		}
	}
	
	public void recalculateSize(Graphics2D g) {
		g.setFont(Config.largeFont);//TODO change back to regularFont!
		int spaceForDiffMarker = 0;
		if (diffStatus != Status.SAME) {
			spaceForDiffMarker = SIZE_OF_PLUS_SIGN+buffer;
		}
		int textWidth = g.getFontMetrics().stringWidth(entity.getName());
		textHeight = g.getFontMetrics().getHeight();
		int cardWidth = Math.max(textWidth+buffer+spaceForDiffMarker,
				MIN_SIZE.width);
		int titleHeight = Math.max(textHeight+buffer, MIN_SIZE.height);
		int cardHeight = titleHeight;
		int x = centerOfGravity.x - cardWidth/2;
		int y = centerOfGravity.y - cardHeight/2;
		textPosition.setLocation((cardWidth-textWidth+spaceForDiffMarker)/2,
							(titleHeight+textHeight/2)/2);
		geometry.setBounds(x, y, cardWidth, cardHeight);			
	}
	
	public void prepareToDraw(Graphics2D g) {
		if (!initialized) {
			initialize(g);
		}
	}
	
	public void draw(Graphics2D g) {
		prepareToDraw(g);
		
		if (flashMode) {
			return;
		}
		
		g.setColor(getCardColor());
		g.setFont(Config.largeFont);//TODO change back to regularFont!
		
		if (ghost) {
			g.drawRoundRect(geometry.x, geometry.y, geometry.width, geometry.height, ROUND_RECT_DIAMETER, ROUND_RECT_DIAMETER);
			g.drawString(entity.getName(), geometry.x+textPosition.x, geometry.y+textPosition.y);
		} else {
			g.fillRoundRect(geometry.x, geometry.y, geometry.width, geometry.height, ROUND_RECT_DIAMETER, ROUND_RECT_DIAMETER);
			g.setColor(Config.textColor);
			g.drawString(entity.getName(), geometry.x+textPosition.x, geometry.y+textPosition.y);
			
			//draw border around card			
			if (diffStatus == Status.SAME) {
				g.setColor(Config.getBorderColorForEntity(entity));
				g.drawRoundRect(geometry.x, geometry.y, geometry.width, geometry.height, ROUND_RECT_DIAMETER, ROUND_RECT_DIAMETER);
			} else {
				
				//if the card's status is NEW, but the entity's status is SAME, that means
				//that the entity is new to the project, but not new to the ward.
				//if the card's status is OLD, but the entity's status is SAME, that means
				//that the entity was removed from the project, but is still in the ward.
				
				Stroke prev = g.getStroke();
				g.setStroke(Config.thickLine);
				if (diffStatus == Status.NEW) {
					if (diffWard.getStatus(entity) == Status.NEW) {
						g.setColor(Config.newEntityColor);
						g.drawRoundRect(geometry.x, geometry.y, geometry.width, geometry.height, ROUND_RECT_DIAMETER, ROUND_RECT_DIAMETER);
						g.setColor(Config.getBorderColorForEntity(entity));
					}
					//draw a plus sign inside the card.
					g.drawLine(geometry.x+textPosition.x-buffer-SIZE_OF_PLUS_SIGN,
							geometry.y+textPosition.y-textHeight/3, geometry.x+textPosition.x-buffer,
							geometry.y+textPosition.y-textHeight/3);
					g.drawLine(geometry.x+textPosition.x-buffer-SIZE_OF_PLUS_SIGN/2,
							geometry.y+textPosition.y-textHeight/3-SIZE_OF_PLUS_SIGN/2,
							geometry.x+textPosition.x-buffer-SIZE_OF_PLUS_SIGN/2,
							geometry.y+textPosition.y-textHeight/3+SIZE_OF_PLUS_SIGN/2);
				} else {
					//Status.OLD
					//draw a minus sign inside the card.
					g.drawLine(geometry.x+textPosition.x-buffer-SIZE_OF_PLUS_SIGN,
							geometry.y+textPosition.y-textHeight/3, geometry.x+textPosition.x-buffer,
							geometry.y+textPosition.y-textHeight/3);
				}
				g.setStroke(Config.thinLine);
				g.setColor(Config.getBorderColorForEntity(entity));
				g.drawRoundRect(geometry.x, geometry.y, geometry.width, geometry.height, ROUND_RECT_DIAMETER, ROUND_RECT_DIAMETER);
				g.setStroke(prev);
			}
		}
		
	}
	
	
	public void drag(final Point delta) {
		geometry.translate(delta.x, delta.y);
		centerOfGravity.translate(delta.x, delta.y);
	}
	
	public void drag(final Point2D.Float delta) {
		geometry.translate((int)delta.x, (int)delta.y);
		centerOfGravity.translate((int)delta.x, (int)delta.y);
	}
	
	public void drawPopupMenu(Component comp, Point pos) {
		//default implementation does nothing
	}
	
	public void actionPerformed(ActionEvent e) {
		String command = e.getActionCommand();
		handleActionEvent(command);
	}
	
	protected void handleActionEvent(String command) {
		if (command.equals(Constants.EDIT_BUTTON_LABEL)) {
			entity.getWard().displayEditSomethingDialog(getEntity());
		}
	}

	
	public String getName() {
		return entity.getName();
	}
	
//	public Family getFamily() {
//		return entity.getFamily();
//	}
	
	public Entity getEntity() {
		return entity;
	}
	
//	public void updateFamilyConnections() {
//		fc.update();
//	}
	
	@Override
	public String toString() {
		return getName();
	}
	
	public int getBottom() {
		return geometry.y+geometry.height;
	}
	
	public int getTop() {
		return geometry.y;
	}
	
	public int getLeft() {
		return geometry.x;
	}
	
	public int getRight() {
		return geometry.x+geometry.width;
	}
	
	public boolean intersects(NameCard other) {
		return (geometry.intersects(other.getBounds()));
	}
	
	public void setDiffWard(DiffWard dw) {
		setDiffWard(dw, dw.getStatus(entity));
	}
	
	/**
	 * This allows you to set the DiffWard and the Status,
	 * in case the default status supplied by the ward is
	 * insufficient. (And it usually is, for the whiteboard.)
	 * @param dw
	 * @param status
	 */
	public void setDiffWard(DiffWard dw, Status status) {
		diffWard = dw;
		diffStatus = status;
		initialized = false;//force a resize before next rendering
	}
	
	public boolean isChanged() {
		return !initialized;
	}
	
	public void setGhostStatus(boolean tf) {
		ghost = tf;
	}
		
	public void toggleFlashMode() {
		flashMode = !flashMode;
	}
	
	public Point getPrevPosition() {
		return prevPosition;
	}

	public void setPrevPosition(Point prevPosition) {
		this.prevPosition.setLocation(prevPosition);
	}

}
